package sample_x5_nj_servlet_ko;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.AbortMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.CompletedMultipartUpload;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.ObjectCannedACL;
import software.amazon.awssdk.services.s3.model.StorageClass;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;
import software.amazon.awssdk.services.s3.presigner.S3Presigner;
import software.amazon.awssdk.services.s3.presigner.model.PresignedUploadPartRequest;

import java.time.Duration;
import java.util.List;
import java.util.Map;

public class AWSS3MultipartUploadHelper {
	
	private final static String BUCKET_NAME = "";
	private final static String ACESSKEY_ID = "";
	private final static String SECRET_ACCESSKEY = "";
	private final static String REGION = "";
	
	private S3Client s3Client;
	private S3Presigner presigner;
	
	AWSS3MultipartUploadHelper() {
		this.s3Client = S3Client.builder()
				.region(Region.AP_NORTHEAST_2)
				.credentialsProvider(StaticCredentialsProvider.create(AwsBasicCredentials.create(ACESSKEY_ID, SECRET_ACCESSKEY)))
				.build();
		this.presigner = S3Presigner.builder()
				.region(Region.AP_NORTHEAST_2)
				.credentialsProvider(StaticCredentialsProvider.create(AwsBasicCredentials.create(ACESSKEY_ID, SECRET_ACCESSKEY)))
				.build();
	}
	
	private StorageClass fromStringToStorageClass(String storageClass) {
		switch (storageClass) {
			case "STANDARD_IA": return StorageClass.STANDARD_IA;
			case "REDUCED_REDUNDANCY": return StorageClass.REDUCED_REDUNDANCY;
			case "ONEZONE_IA": return StorageClass.ONEZONE_IA;
			case "INTELLIGENT_TIERING": return StorageClass.INTELLIGENT_TIERING;
			case "GLACIER": return StorageClass.GLACIER;
			case "DEEP_ARCHIVE": return StorageClass.DEEP_ARCHIVE;
			default: return StorageClass.STANDARD;
		}
	}
	
	private ObjectCannedACL fromStringToCannedACL(String acl) {
		switch (acl) {
			case "public-read": return ObjectCannedACL.PUBLIC_READ;
			case "public-read-write": return ObjectCannedACL.PUBLIC_READ_WRITE;
			case "authenticated-read": return ObjectCannedACL.AUTHENTICATED_READ;
			case "aws-exec-read": return ObjectCannedACL.AWS_EXEC_READ;
			case "bucket-owner-read": return ObjectCannedACL.BUCKET_OWNER_READ;
			case "bucket-owner-full-control": return ObjectCannedACL.BUCKET_OWNER_FULL_CONTROL;
			default: return ObjectCannedACL.PRIVATE;
		}
	}
	
	public String initializeUpload(String key, String storageClass, String acl, String contentType, String metaString) throws JsonMappingException, JsonProcessingException {
		
		ObjectMapper mapper = new ObjectMapper();
		
		Map<String, String> metaMap = mapper.readValue(metaString, new TypeReference<Map<String, String>>() {});
		
		CreateMultipartUploadRequest request = CreateMultipartUploadRequest.builder()
				.bucket(BUCKET_NAME)
				.key(key)				
				.storageClass(fromStringToStorageClass(storageClass))
				.acl(fromStringToCannedACL(acl))
				.contentType(contentType)		
				.metadata(metaMap)
				.build();
		
		CreateMultipartUploadResponse response = s3Client.createMultipartUpload(request);
		
		return String.format("{ \"uid\": \"%s\", \"key\": \"%s\", \"url\": \"https://%s.s3.%s.amazonaws.com/%s\" }", 
				response.uploadId(), 
				response.key(),
				BUCKET_NAME, 
				REGION,
				response.key());
	}

	public String getPresignedURL(String key, String uid, int partNumber) {
		
		UploadPartRequest request = UploadPartRequest.builder()
				.bucket(BUCKET_NAME)
                .key(key)
                .uploadId(uid)
                .partNumber(partNumber)                
                .build();
		
		 PresignedUploadPartRequest presignedRequest = presigner.presignUploadPart(builder -> builder
                .signatureDuration(Duration.ofMinutes(15))
                .uploadPartRequest(request));
		 
		return String.format("{ \"url\": \"%s\" }", presignedRequest.url());
	}

	public String completeUpload(String key, String uid, List<CompletedPart> partETags) {
		
		CompleteMultipartUploadRequest request = CompleteMultipartUploadRequest.builder()
				.bucket(BUCKET_NAME)
                .key(key)
                .uploadId(uid)
                .multipartUpload(CompletedMultipartUpload.builder().parts(partETags).build())
                .build();
		
		CompleteMultipartUploadResponse response = s3Client.completeMultipartUpload(request);
		
		return String.format("{ \"location\": \"%s\" }", response.location());
	}

	public void abortUpload(String key, String uid) {
		
		AbortMultipartUploadRequest request = AbortMultipartUploadRequest.builder()
				.bucket(BUCKET_NAME)
				.key(key)
				.uploadId(uid)
				.build();
		
		s3Client.abortMultipartUpload(request);
	}

}
