﻿using DEXTUpload.NET.Mvc;
using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Text;
using System.Web;
using System.Web.Mvc;

namespace sample_x5_net5_mvc5_en.Controllers
{
    public class FileServiceController : Controller
    {
        [UseDEXT]
        public ActionResult UploadCommon(DEXTUploadX5Vo data)
        {
            var responseData = new StringBuilder();

            foreach (var f in data.Files.Where(f => f != null))
            {
                f.Element.Save();
                responseData.AppendLine(f.Element.LastSavedFileName);
            }

            return Content(responseData.ToString(), "text/plain", Encoding.UTF8);
        }

        [UseDEXT]
        public ActionResult UploadOne(DEXTUploadX5Vo data)
        {
            var element = data.Files[0].Element;
            element.Save();
            return Content(element.LastSavedFileName, "text/plain", Encoding.UTF8);
        }

        [UseDEXT]
        public ActionResult UploadWithMeta(DEXTUploadX5Vo data)
        {
            var responseData = new StringBuilder();

            foreach (var pair in data.Files
                .Zip(data.Metas, (first, second) => new { File = first.Element, Metadata = second })
                .Where(s => s.File != null))
            {
                pair.File.Save();
                responseData.AppendFormat("F:{0}, M:{1}\n", pair.File.FileName, pair.Metadata);
            }

            return Content(responseData.ToString(), "text/plain", Encoding.UTF8);
        }

        [UseDEXT]
        public ActionResult Upload(DEXTUploadX5Vo data)
        {
            var responseData = new StringBuilder();
            var key = null as string;

            foreach (var f in data.Files.Where(f => f != null))
            {
                f.Element.Save();

                // The FileEntity and the FileRepository class are dummy classes used for helping to show the result of uploading.

                key = FileRepository.Add(new FileEntity
                {
                    FileName = f.Element.FileName,
                    SavedFileName = f.Element.LastSavedFileName,
                    MimeType = f.Element.MimeType,
                    Size = f.Element.OriginalFileLength,
                    SavedSize = f.Element.FileLength,
                    Path = f.Element.LastSavedFilePath
                });
                responseData.AppendFormat("{0};", key);
            }

            return Content(responseData.ToString(), "text/plain", Encoding.UTF8);
        }

        [UseDEXT]
        public ActionResult UploadMultiple(DEXTUploadX5Vo data)
        {
            var responseData = new StringBuilder();

            foreach (var pair in data.Files
                .Zip(data.ControlIds, (first, second) => new { File = first.Element, ControlId = second })
                .Zip(data.UniqueIds, (first, second) => new { File = first.File, ControlId = first.ControlId, UniqueId = second  })
                .Where(s => s.File != null))
            {
                pair.File.Save();
                responseData.AppendFormat("{0}|{1}|{2}\n", pair.ControlId, pair.UniqueId, pair.File.FileName);
            }

            return Content(responseData.ToString(), "text/plain", Encoding.UTF8);
        }

        [UseDEXT]
        public ActionResult UploadFolder(DEXTUploadX5Vo data)
        {
            var responseData = new StringBuilder();
            var dir = null as DirectoryInfo;

            foreach (var pair in data.Files.Zip(data.Folders, (first, second) => new { File = first, Folder = second }))
            {
                if (!string.IsNullOrWhiteSpace(pair.Folder))                
                    dir = new DirectoryInfo(Path.Combine(Server.MapPath("~/files/store"), pair.Folder.Substring(1)));
                else
                    dir = new DirectoryInfo(Server.MapPath("~/files/store"));

                if (!dir.Exists) dir.Create();

                if (pair.File != null)
                {
                    pair.File.Element.Save(dir);
                    responseData.AppendFormat("F:{0}\n", pair.File.Element.LastSavedFilePath);
                }
                else
                {
                    responseData.AppendFormat("D:{0}\n", dir);
                }
            }

            return Content(responseData.ToString(), "text/plain", Encoding.UTF8);
        }

        [UseDEXT]
        public ActionResult UploadWithExif(DEXTUploadX5Vo data)
        {
            var responseData = new StringBuilder();
            var tokens = null as string[];
            var strExif = new StringBuilder();

            foreach (var pair in data.Files
                .Zip(data.Exifs, (first, second) => new { File = first.Element, Exif = second })
                .Where(s => s.File != null))
            {
                pair.File.Save();
                tokens = pair.Exif.Split(new string[] { "[SPLT]" }, StringSplitOptions.RemoveEmptyEntries);
                strExif.Clear();
                for (int k = 0, len = tokens.Length; (k + 1) < len; k += 2)
                {
                    strExif.AppendFormat("{0}:{1}\n", tokens[k], tokens[k + 1]);
                }
                responseData.AppendFormat("F:{0}\n{1}\n", pair.File.FileName, strExif.ToString());
            }

            return Content(responseData.ToString(), "text/plain", Encoding.UTF8);
        }

        [UseDEXT]
        public ActionResult UploadExtension(DEXTUploadX5Vo data)
        {
            var key = null as string;

            if (data.Files[0] != null)
            {
                var element = data.Files[0].Element;

                element.Save();

                key = FileRepository.Add(new FileEntity
                {
                    FileName = element.FileName,
                    SavedFileName = element.LastSavedFileName,
                    MimeType = element.MimeType,
                    Size = element.OriginalFileLength,
                    SavedSize = element.FileLength,
                    Path = element.LastSavedFilePath
                });
            }

            return Content(key ?? string.Empty, "text/plain", Encoding.UTF8);
        }

        [AcceptVerbs(HttpVerbs.Head | HttpVerbs.Get)]
        public ActionResult Download(string key)
        {
            var path = null as string;

            if (string.IsNullOrWhiteSpace(key)) throw new HttpException(400, "The key is not set.");
            else if (key.Equals("FID0001")) path = "~/files/attach/bridge_509147.jpg";
            else if (key.Equals("FID0002")) path = "~/files/attach/beach_239826.jpg";
            else if (key.Equals("FID0003")) path = "~/files/attach/cosmos (empty) 195779.jpg";
            else throw new HttpException(404, "The target not found.");

            // If you specify the file name directly, the Content-Dispositon header is set together.
            return File(Server.MapPath(path), "application/octet-stream", Server.UrlEncode(Path.GetFileName(path)).Replace("+", "%20"));
        }

        [HttpGet]
        public ActionResult Open(string key)
        {
            var path = null as string;

            if (string.IsNullOrWhiteSpace(key)) throw new HttpException(400, "The key is not set.");
            else if (key.Equals("FID0001")) path = "~/files/attach/bridge_509147.jpg";
            else if (key.Equals("FID0002")) path = "~/files/attach/beach_239826.jpg";
            else if (key.Equals("FID0003")) path = "~/files/attach/cosmos (empty) 195779.jpg";
            else throw new HttpException(404, "The target not found.");

            return File(Server.MapPath(path), "image/jpeg");
        }

        [HttpPost]
        public ActionResult Compress([Bind(Prefix = "DEXTUploadX5_VIndexes")] string data)
        {
            var vindexes = data.Split(',');
            var targets = new List<string>();

            foreach (string index in vindexes)
            {
                if (index.Equals("IDX0003")) targets.Add(Server.MapPath("~/files/attach/bridge_509147.jpg"));
                else if (index.Equals("IDX0004")) targets.Add(Server.MapPath("~/files/attach/beach_239826.jpg"));
                else if (index.Equals("IDX0005")) targets.Add(Server.MapPath("~/files/attach/cosmos (empty) 195779.jpg"));
            }

            var zipPath = Path.GetTempFileName();

            // Compressed file is created by using ZipArchive supported from .NET Framework 4.5.
            using (var fs = new FileStream(zipPath, FileMode.Create))
            using (var za = new ZipArchive(fs, ZipArchiveMode.Create, false, System.Text.Encoding.UTF8))
            {
                foreach (string path in targets)
                {
                    za.CreateEntryFromFile(path, Path.GetFileName(path));
                }
            }

            // In a real world situation other than the example, 
            // you need to use a different storage method, such as session or database, 
            // so that the path to the compressed file can be obtained from the path where the download should take place.

            FileRepository.Clear();
            var key = FileRepository.Add(new FileEntity
            {
                Path = zipPath,
                MimeType = "application/x-zip-compressed",
                FileName = string.Concat(Path.GetFileNameWithoutExtension(zipPath), ".zip")
            });

            // The path from which the compressed file can be downloaded must be provided in the response data to the client.
            return Content(Url.Action("DownloadZip", new { id = key }), "text/plain", Encoding.UTF8);
        }

        [HttpGet]
        public ActionResult DownloadZip(string id)
        {
            var entry = FileRepository.Get(id);

            if (entry == null)
            {
                throw new HttpException(404, "The zip file not found.");
            }
            else
            {
                return File(entry.Path, "application/x-zip-compressed", entry.FileName);
            }
        }

        [HttpGet]
        public ActionResult GetHDEvent(string action, string key)
        {
            var eventMessage = string.Format("DEXTUploadX5 HD Event {0} = {1}", action, key);

            Debug.WriteLine(eventMessage);

            return Content(eventMessage, "text/plain");
        }
    }
}