﻿namespace sample_x5_net5_en.service
{
	using System;
	using System.Collections.Generic;
	using System.IO;
	using System.Linq;
	using System.Web;
	using System.Net;
	using Amazon;
	using Amazon.S3;
	using Amazon.S3.Model;
	using Newtonsoft.Json;
	using System.Text;

	public class awss3_upload_helper : IHttpHandler
	{

		public class AWSS3MultipartUploadHelper
		{
			public const string ACCESS_KEY = "";
			public const string SECRET_ACCESS_KEY = "";
			public const string REGION = "";
			public const string BUCKET_NAME = "";

			private AmazonS3Client _client;

			public AWSS3MultipartUploadHelper()
			{
				_client = new AmazonS3Client(ACCESS_KEY, SECRET_ACCESS_KEY, RegionEndpoint.GetBySystemName(REGION));
			}

			private S3CannedACL FromStringToCannedACL(string acl)
			{
				switch (acl)
				{
					case "public-read": return S3CannedACL.PublicRead;
					case "public-read-write": return S3CannedACL.PublicReadWrite;
					case "authenticated-read": return S3CannedACL.AuthenticatedRead;
					case "aws-exec-read": return S3CannedACL.AWSExecRead;
					case "bucket-owner-read": return S3CannedACL.BucketOwnerRead;
					case "bucket-owner-full-control": return S3CannedACL.BucketOwnerFullControl;
					default: return S3CannedACL.Private;
				}
			}

			private S3StorageClass FromStringToStorageClass(string storageClass)
			{
				switch (storageClass)
				{
					case "STANDARD_IA": return S3StorageClass.StandardInfrequentAccess;
					case "REDUCED_REDUNDANCY": return S3StorageClass.ReducedRedundancy;
					case "ONEZONE_IA": return S3StorageClass.OneZoneInfrequentAccess;
					case "INTELLIGENT_TIERING": return S3StorageClass.IntelligentTiering;
					case "GLACIER": return S3StorageClass.Glacier;
					case "DEEP_ARCHIVE": return S3StorageClass.DeepArchive;
					default: return S3StorageClass.Standard;
				}
			}

			public string InitializeUpload(string key, string storageClass, string acl, string contentType, string metaString)
			{
				var request = new InitiateMultipartUploadRequest
				{
					BucketName = BUCKET_NAME,
					Key = key,
					ContentType = contentType,
					StorageClass = FromStringToStorageClass(storageClass),
					CannedACL = FromStringToCannedACL(acl)
				};

				var dictionary = JsonConvert.DeserializeObject<Dictionary<string, string>>(metaString);
				var list = new List<KeyValuePair<string, string>>(dictionary);
				foreach (var pair in list)
				{
					request.Metadata.Add(pair.Key, pair.Value);
				}

				InitiateMultipartUploadResponse initiatedResponse = _client.InitiateMultipartUpload(request);

				return string.Format("{{ \"uid\": \"{0}\", \"key\": \"{1}\", \"url\": \"https://{2}.s3.{3}.amazonaws.com/{1}\" }}", initiatedResponse.UploadId, initiatedResponse.Key, BUCKET_NAME, REGION);
			}

			public string GetPresignedURL(string key, string uid, int partNumber)
			{
				return string.Format("{{ \"url\": \"{0}\" }}", _client.GetPreSignedURL(new GetPreSignedUrlRequest
				{
					BucketName = BUCKET_NAME,
					Key = key,
					UploadId = uid,
					PartNumber = partNumber,
					Verb = HttpVerb.PUT,
					Expires = DateTime.UtcNow.AddMinutes(15)
				}));
			}

			public string CompleteUpload(string key, string uid, List<PartETag> partETags)
			{
				CompleteMultipartUploadResponse completedResponse = _client.CompleteMultipartUpload(new CompleteMultipartUploadRequest
				{
					BucketName = BUCKET_NAME,
					Key = key,
					UploadId = uid,
					PartETags = partETags
				});

				return string.Format("{{ \"location\": \"{0}\" }}", completedResponse.Location);
			}

			public void AbortUpload(string key, string uid)
			{
				_client.AbortMultipartUpload(new AbortMultipartUploadRequest
				{
					BucketName = BUCKET_NAME,
					Key = key,
					UploadId = uid
				});
			}
		}

		private void ProcessStepZero(HttpRequest req, HttpResponse res)
		{
			var key = req.Form["key"];
			var acl = req.Form["acl"] ?? "private";
			var sclass = req.Form["sclass"] ?? "STANDARD";
			var type = req.Form["type"];
			var meta = req.Form["metadata"];

			if (!string.IsNullOrEmpty(key) && !string.IsNullOrEmpty(type))
			{
				var helper = new AWSS3MultipartUploadHelper();
				var responseString = helper.InitializeUpload(key, sclass, acl, type, meta);
				res.Write(responseString);
			}
			else
			{
				res.StatusCode = 400;
				res.Write("{ \"error\": \"Invalid parameters.\" }");
			}
		}

		private void ProcessStepOne(HttpRequest req, HttpResponse res)
		{
			var key = req.Form["key"];
			var uid = req.Form["uid"];
			var pnum = req.Form["pnum"];
			var partNumber = 0;

			if (!string.IsNullOrEmpty(key) && !string.IsNullOrEmpty(uid) && int.TryParse(pnum, out partNumber))
			{
				var helper = new AWSS3MultipartUploadHelper();
				var responseString = helper.GetPresignedURL(key, uid, partNumber);
				res.Write(responseString);
			}
			else
			{
				res.StatusCode = 400;
				res.Write("{ \"error\": \"Invalid parameters.\" }");
			}
		}

		private void ProcessStepTwo(HttpRequest req, HttpResponse res)
		{
			var key = req.Form["key"];
			var uid = req.Form["uid"];
			var pnums = req.Form.GetValues("pnum");
			var etags = req.Form.GetValues("etag");
			var num = 0;
			var zipped = pnums.Zip(etags, (ns, et) => int.TryParse(ns, out num) ? new PartETag { PartNumber = num, ETag = et } : null).Where(pe => pe != null);

			if (!string.IsNullOrEmpty(key) && !string.IsNullOrEmpty(uid) && zipped.Count() > 0)
			{
				var helper = new AWSS3MultipartUploadHelper();
				var responseString = helper.CompleteUpload(key, uid, zipped.ToList());
				res.Write(responseString);
			}
			else
			{
				res.StatusCode = 400;
				res.Write("{ \"error\": \"Invalid parameters.\" }");
			}
		}

		private void ProcessStepThree(HttpRequest req, HttpResponse res)
		{
			var key = req.Form["key"];
			var uid = req.Form["uid"];

			if (!string.IsNullOrEmpty(key) && !string.IsNullOrEmpty(uid))
			{
				var helper = new AWSS3MultipartUploadHelper();
				helper.AbortUpload(key, uid);
			}
			else
			{
				res.StatusCode = 400;
				res.Write("{ \"error\": \"Invalid parameters.\" }");
			}
		}

		public void ProcessRequest(HttpContext context)
		{
			var req = context.Request;
			var res = context.Response;
			var step = req.QueryString["step"] ?? string.Empty;

			res.ContentType = "application/json";

			try
			{
				switch (step)
				{
					case "0": ProcessStepZero(req, res); break;
					case "1": ProcessStepOne(req, res); break;
					case "2": ProcessStepTwo(req, res); break;
					case "3": ProcessStepThree(req, res); break;
					default:
						res.StatusCode = 400;
						res.Write(string.Format("{ \"error\": \"'{0}' is an unknown step.\" }", step));
						break;
				}
			}
			catch (Exception ex)
			{
				res.StatusCode = 500;
				res.Write(string.Format("{{ \"error\": \"{0}\" }}", ex.Message));
			}
		}

		public bool IsReusable
		{
			get
			{
				return false;
			}
		}
	}
}