package sample_x5_jk_boot_gradle_jar_ko;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import jakarta.servlet.ServletContext;
import jakarta.servlet.http.HttpServletRequest;

import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.context.ServletContextAware;
import org.springframework.web.server.ResponseStatusException;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.support.ServletUriComponentsBuilder;

import dextuploadjk.engine.CompressUtil;
import dextuploadjk.engine.FileItem;
import dextuploadjk.engine.FileResponseContentDisposition;
import dextuploadjk.engine.FileSaveOption;
import dextuploadjk.support.spring.JKFileDownloadView;

@Controller
public class FileServiceController implements ServletContextAware {
	
	private ServletContext servletContext;

	@Override
	public void setServletContext(ServletContext servletContext) {
		this.servletContext = servletContext;
	}
	
	@PostMapping(value = { "/upload-simple", "/upload-oraf" }, produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String uploadSimple(
			// DEXTUploadX5List 클래스는 제품 라이브러리에 포함된 클래스가 아니며, 
			// ArgumentResolver 기능을 사용하여 DEXTUploadX5로부터 전달되는 폼 정보를 가지는 VO 클래스이다.
			DEXTUploadX5List list) throws IOException {
		
		FileItem fileItem = null;
		List<String> data = new ArrayList<>();
		
		for (DEXTUploadX5Item item : list) {
			fileItem = (FileItem)item.getFileData();
			if (fileItem.isEmpty()) continue;
			// 대상이 올바른 파일이라면 실제 저장할 위치로 임시 파일을 저장(복사 혹은 이동)한다.
			// 인자로 주어진 디렉터리 경로가 없다면 JKSpringExtensionUploadFilter 필터의 defaultRepository 위치에 저장된다.
			fileItem.save();
			// 저장된 파일 이름을 응답 데이터에 기록하여 업로드가 잘 됐는지 확인할 수 있도록 한다.
			data.add(fileItem.getFilename());
		}
		
		return String.join("\n", data);
	}
	
	@PostMapping(value = "/upload-metadata", produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String uploadMetadata(DEXTUploadX5List list) {
		
		FileItem fileItem = null;
		List<String> data = new ArrayList<>();
		String metadata = null;
		
		for (DEXTUploadX5Item item : list) {
			fileItem = (FileItem)item.getFileData();	
			if (fileItem.isEmpty()) continue;
			fileItem.save();
			// 메타데이터 문자열 정보를 얻는다.
			metadata = item.getMetaData();
			data.add(String.format("%s, %s", fileItem.getFilename(), metadata));
		}
		
		return String.join("\n", data);
	}
	
	@PostMapping(value = "/upload-orof", produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String uploadOROF(
			// DEXTUploadX5Item 클래스는 제품 라이브러리에 포함된 클래스가 아니며, 
			// ArgumentResolver 기능을 사용하여 DEXTUploadX5로부터 전달되는 폼 정보를 가지는 VO 클래스이다.
			DEXTUploadX5Item item) throws IOException {
		
		FileItem fileItem = (FileItem)item.getFileData();
		
		if (fileItem.isEmpty()) return "- no file -";
		
		fileItem.save();
		
		return fileItem.getFilename();
	}
	
	@PostMapping(value = "/upload-files", produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String uploadFiles(DEXTUploadX5List list) {
		
		// DB에 업로드된 파일 정보를 저장하는 서비스를 담당하는 클래스라고 생각하자.
		FakeFileService.clearSession();
		
		FileItem fileItem = null;
		List<String> fileKeys = new ArrayList<>();
		String key = null;
		
		for (DEXTUploadX5Item item : list) {
			fileItem = (FileItem)item.getFileData();
			if (fileItem.isEmpty()) continue;
			fileItem.save();
			
			// 저장된 파일 정보를 DB에 기록한다.
			FakeFile ff = new FakeFile();
			ff.setFormName(fileItem.getFieldName());
			ff.setFileName(fileItem.getFilename());
			ff.setSize(fileItem.getFileSize());
			ff.setMimeType(fileItem.getContentType());
			ff.setPath(fileItem.getLastSavedFilePath());
			// 저장된 파일의 키를 얻는다.
			key = FakeFileService.add(ff);			
			fileKeys.add(key);
		}
		// 구분자(;)를 구분된 파일의 키 값들을 응답 데이터로 기록한다. 
		return String.join(";", fileKeys);
	}
	
	@PostMapping(value = "/upload-multiple", produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String uploadMultiple(DEXTUploadX5List list) {
		
		List<String> data = new ArrayList<>();
		FileItem fileItem = null;
		String controlId = null;
		String uniqueId = null;
		
		for (DEXTUploadX5Item item : list) {
			fileItem = (FileItem)item.getFileData();
			if (fileItem.isEmpty()) continue;
			fileItem.save();
			// 컴포넌트 아이디, 파일 항목 아이디를 얻는다.
			controlId = item.getControlId();
			uniqueId = item.getUniqueId();
			data.add(String.format("%s|%s|%s", controlId, uniqueId, fileItem.getFilename()));
		}
		
		return String.join("\n", data);
	}
	
	@PostMapping(value = "/upload-folder", produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String uploadFolder(DEXTUploadX5List list) throws IOException {
		
		FileItem fileItem = null;
		List<String> data = new ArrayList<>();
		String sub = null;
		File dir = null;
		
		for (DEXTUploadX5Item item : list) {
			fileItem = (FileItem)item.getFileData();
			if (fileItem.isEmpty()) continue;
			
			// 파일의 폴더 경로를 얻은 후, 저장 디렉터리 하위에 하위 폴더 경로를 생성한다.
			sub = item.getFolder();
			dir = new File(fileItem.getEnviroment().getDefaultRepository(), sub);
			if (dir.exists() == false) dir.mkdirs();
			
			// 대상 폴더로 저장한다.
			FileSaveOption option = new FileSaveOption();
			option.setTargetDirectoryPath(dir.getCanonicalPath());
			fileItem.save(option);
			data.add(fileItem.getLastSavedFilePath());
		}
		
		return String.join("\n", data);
	}
	
	@PostMapping(value = "/upload-exif", produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String uploadExif(DEXTUploadX5List list) {
		
		FileItem fileItem = null;
		List<String> data = new ArrayList<>();
		String exif = null;
		
		for (DEXTUploadX5Item item : list) {
			fileItem = (FileItem)item.getFileData();	
			if (fileItem.isEmpty()) continue;
			fileItem.save();
			// EXIF 문자열 정보를 얻는다.
			exif = item.getEXIFData();
			data.add(String.format("%s => %s", fileItem.getFilename(), exif));
		}
		
		return String.join("\n", data);
	}
	
	@PostMapping(value = "/upload-extension", produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String uploadExtension(DEXTUploadX5Item item) {
		
		FileItem fileItem = (FileItem)item.getFileData();
		if (fileItem.isEmpty()) return "- no file -";
		
		fileItem.save();
		
		FakeFile ff = new FakeFile();
		ff.setFormName(fileItem.getFieldName());
		ff.setFileName(fileItem.getFilename());
		ff.setSize(fileItem.getFileSize());
		ff.setMimeType(fileItem.getContentType());
		ff.setPath(fileItem.getLastSavedFilePath());		
		
		return FakeFileService.add(ff);
	}
	
	@GetMapping("/download-file")
	public ModelAndView downloadFile(@RequestParam(value = "key") String key) {
		File target = null;
		
		String fileRoot = servletContext.getRealPath("/files/attach");
		
		if (key.equals("F0001")) target = new File(fileRoot, "서강대교_509147.jpg");
		else if (key.equals("F0002")) target = new File(fileRoot, "우도해변_239826.jpg");
		else if (key.equals("F0003")) target = new File(fileRoot, "코스모스 (빈공간) 195779.jpg");
		
		if (target == null  || target.exists() == false || target.isFile() == false) {
			throw new ResponseStatusException(HttpStatus.NOT_FOUND, "주어진 키에 해당하는 파일 정보가 없습니다.");
		}
		
		// 파일을 다운로드 하기 위해서 JKFileDownloadView 객체를 생성한다.
		JKFileDownloadView view = new JKFileDownloadView(target);
		view.setCharsetName("UTF-8");
		// DEXTUploadX5를 사용하여 다운로드할 때는 Attachment로 설정하는 것이 좋다.
		view.setContentDisposition(FileResponseContentDisposition.Attachment);
		view.setAllowingWeakRange(true);
		// 다운로드가 시작된다.
		return new ModelAndView(view);
	}
	
	@GetMapping("/open-file")
	public ModelAndView openFile(@RequestParam(value = "key") String key) {		
		File target = null;
		
		String fileRoot = servletContext.getRealPath("/files/attach");
		
		if (key.equals("F0001")) target = new File(fileRoot, "서강대교_509147.jpg");
		else if (key.equals("F0002")) target = new File(fileRoot, "우도해변_239826.jpg");
		else if (key.equals("F0003")) target = new File(fileRoot, "코스모스 (빈공간) 195779.jpg");
		
		if (target == null  || target.exists() == false || target.isFile() == false) {
			throw new ResponseStatusException(HttpStatus.NOT_FOUND, "주어진 키에 해당하는 파일 정보가 없습니다.");
		}
		
		JKFileDownloadView view = new JKFileDownloadView(target);
		view.setCharsetName("UTF-8");
		// DEXTUploadX5를 사용하여 파일을 열 때는 Inline으로 설정해야 한다.
		view.setContentDisposition(FileResponseContentDisposition.Inline);
		// 브라우저에서 파일을 열기 위해서는 올바른 MIME 타입을 요구하는 경우가 있으므로 대상의 MIME-TYPE을 지정해주자.
		view.setMime("image/jpeg");
		view.setAllowingWeakRange(true);

		return new ModelAndView(view);
	}
	
	@PostMapping(value = "/compress-zip", produces = "text/plain;charset=UTF-8")
	@ResponseBody
	public String makeZip(@RequestParam(value = "DEXTUploadX5_VIndexes") String vindexes, HttpServletRequest request) {
		
		String fileRoot = servletContext.getRealPath("/files");
		
		List<File> files = new ArrayList<>();
		String[] tokens = vindexes.split(",");
		for (int i = 0; i < tokens.length; i++) {
			if (tokens[i].equals("IDX0003")) files.add(new File(fileRoot, "attach/서강대교_509147.jpg"));
			if (tokens[i].equals("IDX0004")) files.add(new File(fileRoot, "attach/우도해변_239826.jpg"));
			if (tokens[i].equals("IDX0005")) files.add(new File(fileRoot, "attach/코스모스 (빈공간) 195779.jpg"));
		}
		
		// 임시 위치에 압축 파일을 생성한다.
		CompressUtil cu = new CompressUtil();
		File zipped = cu.zip(files, new File(fileRoot, "/temp"), "UTF-8", false, false);
		
		FakeFile target = new FakeFile();
		target.setMimeType("application/x-zip-compressed");
		target.setFileName(zipped.getName());
		target.setPath(zipped.getAbsolutePath());
		// 압축 파일 정보를 DB에 기록하고 그 키 값을 얻는다.
		String compresskey = FakeFileService.add(target);
		
		return ServletUriComponentsBuilder
		        .fromRequestUri(request)
		        .replacePath("/download-zip")
		        .queryParam("compresskey", compresskey)
		        .build()
		        .toUriString();
	}
	
	@GetMapping("/download-zip")
	public ModelAndView downloadZip(@RequestParam(value = "compresskey") String key) {
		
		FakeFile target = FakeFileService.get(key);
		
		if (target != null) {
			JKFileDownloadView view = new JKFileDownloadView();
			view.setFile(new File(target.getPath()));
			view.setCharsetName("UTF-8");
			// 만들어진 압축 파일 이름을 DEXTUploadX5는 모르기 때문에, 파일 이름을 그대로 가져다 쓰기 위해서, AttachmentWithName을 설정한다.
			view.setContentDisposition(FileResponseContentDisposition.AttachmentWithName);
			view.setAllowingWeakRange(false);
			view.setRemoveAfterDownloading(true);
			return new ModelAndView(view);			
		} else {
			throw new ResponseStatusException(HttpStatus.NOT_FOUND);
		}
	}	
	
}
